/* opiesubr.c: OTP support subroutines.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

   History:

   Modified at NRL for OPIE 2.01: opiereadpass() -- Before messing with
                 terminal attributes, make sure that we're actually on a
                 terminal in order to be more pipe-friendly.
        
   Modified at NRL for OPIE 2.0. Changes include making the code more
      generic by changing it to support both MD4 and MD5 with the
      default set at compile time using the MDX symbol and cleaned
      includes and code to be POSIX compliant and hence portable to
      more systems.
   Modified at NRL to support MD5 as an alternative to MD4 for OPIE 1.0.
   Written at Bellcore for the S/Key Version 1 software distribution
      (skeysubr.c).
*/
#include "opie_cfg.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>  /* ANSI C standard library */

void opiesevenbit(char *s);

void getutmpentry __ARGS((char *, struct UTMPX *));


#include "global.h"
#include "opie.h"
#include "md5.h"
#include "md4.h"

/* Crunch a key:
 * concatenate the seed and the password, run through MD4 or MD5 and
 * collapse to 64 bits. This is defined as the user's starting key.
 */
int opiekeycrunch(unsigned algorithm, char *result, char *seed, char *passwd)
{
  char *buf;
  struct MD5_CTX md5;
  struct MD4_CTX md4;
  unsigned long results[4];
  unsigned int buflen;

/* NOTE - we need to bound seed and password before this point.
   "any length" can bite us... */
  buflen = strlen(seed) + strlen(passwd);
  if ((buf = (char *)malloc(buflen + 1)) == NULL)
    return -1;
  strcpy(buf, seed);
  strcat(buf, passwd);

  /* Crunch the key through MD[45] */
  opiesevenbit(buf);
  if (4 == algorithm) {
    MD4Init(&md4);
    MD4Update(&md4, (unsigned char *) buf, buflen);
    MD4Final((unsigned char *) results, &md4);
  } else {
    MD5Init(&md5);
    MD5Update(&md5, (unsigned char *) buf, buflen);
    MD5Final((unsigned char *) results, &md5);
  }
  free(buf);

  results[0] ^= results[2];
  results[1] ^= results[3];

  memcpy(result, (char *) results, 8);
  return 0;
}

/*
 * The one-way function f(x).
 * Takes 8 bytes and returns 8 bytes in place.
 * The value of "algorithm" determines whether MD4 or MD5 is used,
 * where (algorithm==5) implies MD5 and (algorithm==4) implies MD4.
 */
void opiehash(char *x, unsigned algorithm)
{
  struct MD5_CTX md5;
  struct MD4_CTX md4;
  unsigned long results[4];

  if (4 == algorithm) {
    MD4Init(&md4);
    MD4Update(&md4, (unsigned char *) x, 8);
    MD4Final((unsigned char *) results, &md4);
  } else {
    /* default algorithm is MD5 */
    MD5Init(&md5);
    MD5Update(&md5, (unsigned char *) x, 8);
    MD5Final((unsigned char *) results, &md5);
  }
  /* Fold 128 to 64 bits */
  results[0] ^= results[2];
  results[1] ^= results[3];

  /* Only works on byte-addressed little-endian machines!! */
  memcpy(x, (char *) results, 8);
}


/* Strip trailing cr/lf from a line of text */
void opiestrip_crlf(char *buf)
{
  char *cp;

  if ((cp = strchr(buf, '\r')) != NULL)
    *cp = '\0';

  if ((cp = strchr(buf, '\n')) != NULL)
    *cp = '\0';
}

/************************/



void opiesevenbit(char *s)
{
  /* make sure there are only 7 bit code in the line */
  while (*s) {
    *s = 0x7f & (*s);
    s++;
  }
}

void opieversion()
{
  exit(0);
}



/* 
   Applies "good password" rules to the secret pass phrase.

   We currently implement the following:

   Passwords must be at least OPIE_PASS_MIN (10) characters long.
   Passwords must be at most OPIE_PASS_MAX (127) characters long.

   N.B.: Passing NULL pointers to this function is a bad idea.
*/
int opiepasscheck(char *passphrase)
{
   int len = strlen(passphrase);

   if (len < OPIE_PASS_MIN)
      return 1;

   if (len > OPIE_PASS_MAX)
      return 1;

   return 0;
}
